# SPDX-License-Identifier: GPL-3.0-or-later
"""
CGMist RenderCheck - Blender File Validator for Render Farms
Validates blend files for common issues before submitting to render farms.

Compatible with Blender 3.0 - 5.0+
"""

bl_info = {
    "name": "CGMist RenderCheck",
    "author": "CGMist Team",
    "version": (2, 0, 0),
    "blender": (3, 0, 0),
    "location": "View3D > Sidebar > CGMist",
    "description": "Validates blend files for render farm compatibility with smart deduplication",
    "warning": "",
    "doc_url": "https://www.cgmist.com/docs/rendercheck",
    "tracker_url": "https://www.cgmist.com/support",
    "category": "Render",
}

import bpy
from bpy.props import CollectionProperty, StringProperty, IntProperty, EnumProperty, BoolProperty
from bpy.types import PropertyGroup

from . import validator
from . import fixer
from . import ui


# -----------------------------------------------------------------------------
# Blender Version Compatibility Helpers
# -----------------------------------------------------------------------------

def get_blender_version():
    """Get Blender version as a tuple for comparison."""
    return bpy.app.version


def is_blender_4_plus():
    """Check if running Blender 4.0 or higher."""
    return bpy.app.version >= (4, 0, 0)


def is_blender_3_6_plus():
    """Check if running Blender 3.6 or higher."""
    return bpy.app.version >= (3, 6, 0)


# -----------------------------------------------------------------------------
# Property Groups for storing validation results
# -----------------------------------------------------------------------------

class CGMIST_PG_Issue(PropertyGroup):
    """Property group for a single validation issue."""
    severity: EnumProperty(
        name="Severity",
        items=[
            ('CRITICAL', "Critical", "Will cause render failure"),
            ('HIGH', "High", "Likely to cause problems"),
            ('MEDIUM', "Medium", "May affect quality or performance"),
            ('LOW', "Low", "Minor optimization suggestion"),
        ],
        default='LOW'
    )
    issue_type: StringProperty(name="Issue Type", default="")
    title: StringProperty(name="Title", default="")
    description: StringProperty(name="Description", default="")
    fix_instructions: StringProperty(name="Fix Instructions", default="")
    data_name: StringProperty(name="Data Name", default="")
    data_type: StringProperty(name="Data Type", default="")
    can_auto_fix: BoolProperty(name="Can Auto Fix", default=False)
    extra_data: StringProperty(name="Extra Data", default="")  # JSON string for additional data


class CGMIST_PG_Results(PropertyGroup):
    """Property group for storing all validation results."""
    issues: CollectionProperty(type=CGMIST_PG_Issue)
    active_issue_index: IntProperty(name="Active Issue", default=0)
    validation_status: StringProperty(name="Status", default="NOT_RUN")
    critical_count: IntProperty(name="Critical", default=0)
    high_count: IntProperty(name="High", default=0)
    medium_count: IntProperty(name="Medium", default=0)
    low_count: IntProperty(name="Low", default=0)
    total_count: IntProperty(name="Total", default=0)
    last_scan_time: StringProperty(name="Last Scan", default="")


# -----------------------------------------------------------------------------
# Operators
# -----------------------------------------------------------------------------

class CGMIST_OT_OpenWebsite(bpy.types.Operator):
    """Open CGMist website in browser"""
    bl_idname = "cgmist.open_website"
    bl_label = "Visit CGMist.com"
    bl_description = "Open CGMist website for high-quality render farm services"

    def execute(self, context):
        import webbrowser
        webbrowser.open("https://www.cgmist.com")
        return {'FINISHED'}


class CGMIST_OT_Validate(bpy.types.Operator):
    """Scan the current scene for render farm compatibility issues"""
    bl_idname = "cgmist.validate"
    bl_label = "Scan Scene"
    bl_description = "Validate the current blend file for render farm compatibility"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        try:
            from datetime import datetime

            # Run validation
            results = validator.run_all_checks()

            # Store results in scene properties
            render_check = context.scene.cgmist_results
            render_check.issues.clear()

            # Process each severity level
            severity_map = {
                'critical': 'CRITICAL',
                'high': 'HIGH',
                'medium': 'MEDIUM',
                'low': 'LOW'
            }

            for severity_key, severity_enum in severity_map.items():
                for issue in results['issues'].get(severity_key, []):
                    item = render_check.issues.add()
                    item.severity = severity_enum
                    item.issue_type = issue.get('issue', 'unknown')
                    item.title = self._get_issue_title(issue)
                    item.description = issue.get('user_message', '')
                    item.fix_instructions = issue.get('fix_instructions', '')
                    item.data_name = issue.get('name', issue.get('material', issue.get('object', '')))
                    item.data_type = issue.get('type', '')
                    item.can_auto_fix = issue.get('can_auto_fix', False)
                    
                    # Store extra data for smart deduplication
                    if 'extra_data' in issue:
                        import json
                        item.extra_data = json.dumps(issue['extra_data'])

            # Update summary counts
            summary = results.get('summary', {})
            render_check.critical_count = summary.get('critical', 0)
            render_check.high_count = summary.get('high', 0)
            render_check.medium_count = summary.get('medium', 0)
            render_check.low_count = summary.get('low', 0)
            render_check.total_count = summary.get('total_issues', 0)
            render_check.validation_status = results.get('validation_status', 'UNKNOWN')
            render_check.last_scan_time = datetime.now().strftime("%H:%M:%S")

            # Report to user
            if render_check.validation_status == 'PASS':
                self.report({'INFO'}, "CGMist RenderCheck: Validation passed - no issues found!")
            elif render_check.validation_status == 'WARNING':
                self.report({'WARNING'}, f"CGMist RenderCheck: Found {render_check.total_count} issue(s) - review recommended")
            else:
                self.report({'ERROR'}, f"CGMist RenderCheck: Found {render_check.critical_count} critical issue(s) that must be fixed")

            return {'FINISHED'}

        except Exception as e:
            self.report({'ERROR'}, f"CGMist RenderCheck: Validation failed: {str(e)}")
            return {'CANCELLED'}

    def _get_issue_title(self, issue):
        """Generate a human-readable title for an issue."""
        issue_type = issue.get('issue', '')
        name = issue.get('name', issue.get('material', issue.get('texture', issue.get('object', ''))))

        titles = {
            'unpacked_image': f"Unpacked Image: {name}",
            'unpacked_movie_clip': f"Unpacked Movie Clip: {name}",
            'unpacked_sound': f"Unpacked Sound: {name}",
            'unpacked_volume': f"Unpacked Volume: {name}",
            'unpacked_font': f"Unpacked Font: {name}",
            'unpacked_library': f"Unpacked Library: {name}",
            'missing_texture': f"Missing Texture in: {issue.get('material', 'Unknown')}",
            'psd_file_detected': f"PSD File: {issue.get('texture', name)}",
            'large_texture': f"Large Texture ({issue.get('category', '')}): {issue.get('texture', name)}",
            'duplicate_materials': f"Duplicate Material: {issue.get('base_material', name)}",
            'duplicate_materials_identical': f"Identical Materials (Safe): {issue.get('base_material', name)}",
            'duplicate_materials_conflict': f"⚠ Naming Conflict: {issue.get('base_material', name)}",
            'duplicate_images': f"Duplicate Images: {name}",
            'zero_scale_object': f"Zero-Scale Object: {issue.get('object', name)}",
            'unconnected_texture_node': f"Unused Texture Node: {issue.get('material', 'Unknown')}",
            'unapplied_modifier': f"Unapplied Modifier: {issue.get('object', 'Unknown')}",
        }

        return titles.get(issue_type, f"{issue_type}: {name}")


class CGMIST_OT_AutoFixAll(bpy.types.Operator):
    """Attempt to automatically fix all detected issues"""
    bl_idname = "cgmist.autofix_all"
    bl_label = "Auto-Fix All"
    bl_description = "Attempt to automatically fix all safe, fixable issues"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        fixed_count = 0
        error_count = 0
        messages = []

        # Run all fixers
        fix_functions = [
            ("Pack Resources", fixer.fix_pack_resources),
            ("Relative Paths", fixer.fix_relative_paths),
            ("Render Settings", fixer.fix_render_settings),
            ("Duplicate Images", fixer.fix_duplicate_images),
            ("Identical Materials", fixer.fix_identical_materials),
        ]

        for name, fix_func in fix_functions:
            try:
                success, message = fix_func()
                if success:
                    fixed_count += 1
                    messages.append(f"{name}: OK")
                else:
                    error_count += 1
                    messages.append(f"{name}: {message}")
            except Exception as e:
                error_count += 1
                messages.append(f"{name}: Error - {str(e)}")

        # Re-run validation to update results
        bpy.ops.cgmist.validate()

        # Report summary
        if error_count == 0:
            self.report({'INFO'}, f"CGMist Auto-fix complete: {fixed_count} fixes applied")
        else:
            self.report({'WARNING'}, f"CGMist Auto-fix: {fixed_count} succeeded, {error_count} failed")

        return {'FINISHED'}


class CGMIST_OT_FixSingle(bpy.types.Operator):
    """Fix a specific issue"""
    bl_idname = "cgmist.fix_single"
    bl_label = "Fix Issue"
    bl_description = "Attempt to fix this specific issue"
    bl_options = {'REGISTER', 'UNDO'}

    issue_index: IntProperty(default=-1)

    def execute(self, context):
        render_check = context.scene.cgmist_results

        if self.issue_index < 0 or self.issue_index >= len(render_check.issues):
            self.report({'ERROR'}, "Invalid issue index")
            return {'CANCELLED'}

        issue = render_check.issues[self.issue_index]

        try:
            # Parse extra data if available
            extra_data = None
            if issue.extra_data:
                import json
                try:
                    extra_data = json.loads(issue.extra_data)
                except:
                    pass

            success, message = fixer.fix_single_issue(
                issue.issue_type, 
                issue.data_name, 
                issue.data_type,
                extra_data
            )

            if success:
                self.report({'INFO'}, f"CGMist: Fixed - {issue.title}")
                # Re-run validation
                bpy.ops.cgmist.validate()
            else:
                self.report({'WARNING'}, f"CGMist: Could not fix - {message}")

            return {'FINISHED'}

        except Exception as e:
            self.report({'ERROR'}, f"CGMist: Fix failed - {str(e)}")
            return {'CANCELLED'}


class CGMIST_OT_PreflightDialog(bpy.types.Operator):
    """Open the CGMist Pre-flight Check dialog"""
    bl_idname = "cgmist.preflight_dialog"
    bl_label = "CGMist Pre-flight Check"
    bl_description = "Open the full pre-flight validation report in a popup window"
    bl_options = {'REGISTER'}

    def execute(self, context):
        return {'FINISHED'}

    def invoke(self, context, event):
        # Run validation directly (not via operator) to avoid RuntimeError from report()
        self._run_validation(context)
        # Open as popup dialog
        return context.window_manager.invoke_props_dialog(self, width=700)
    
    def _run_validation(self, context):
        """Run validation and store results without using operator report()."""
        try:
            from datetime import datetime
            
            # Run validation
            results = validator.run_all_checks()

            # Store results in scene properties
            render_check = context.scene.cgmist_results
            render_check.issues.clear()

            # Process each severity level
            severity_map = {
                'critical': 'CRITICAL',
                'high': 'HIGH',
                'medium': 'MEDIUM',
                'low': 'LOW'
            }

            for severity_key, severity_enum in severity_map.items():
                for issue in results['issues'].get(severity_key, []):
                    item = render_check.issues.add()
                    item.severity = severity_enum
                    item.issue_type = issue.get('issue', 'unknown')
                    item.title = self._get_issue_title(issue)
                    item.description = issue.get('user_message', '')
                    item.fix_instructions = issue.get('fix_instructions', '')
                    item.data_name = issue.get('name', issue.get('material', issue.get('object', '')))
                    item.data_type = issue.get('type', '')
                    item.can_auto_fix = issue.get('can_auto_fix', False)
                    
                    # Store extra data for smart deduplication
                    if 'extra_data' in issue:
                        import json
                        item.extra_data = json.dumps(issue['extra_data'])

            # Update summary counts
            summary = results.get('summary', {})
            render_check.critical_count = summary.get('critical', 0)
            render_check.high_count = summary.get('high', 0)
            render_check.medium_count = summary.get('medium', 0)
            render_check.low_count = summary.get('low', 0)
            render_check.total_count = summary.get('total_issues', 0)
            render_check.validation_status = results.get('validation_status', 'UNKNOWN')
            render_check.last_scan_time = datetime.now().strftime("%H:%M:%S")

        except Exception as e:
            print(f"CGMist RenderCheck: Validation failed: {str(e)}")

    def _get_issue_title(self, issue):
        """Generate a human-readable title for an issue."""
        issue_type = issue.get('issue', '')
        name = issue.get('name', issue.get('material', issue.get('texture', issue.get('object', ''))))

        titles = {
            'unpacked_image': f"Unpacked Image: {name}",
            'unpacked_movie_clip': f"Unpacked Movie Clip: {name}",
            'unpacked_sound': f"Unpacked Sound: {name}",
            'unpacked_volume': f"Unpacked Volume: {name}",
            'unpacked_font': f"Unpacked Font: {name}",
            'unpacked_library': f"Unpacked Library: {name}",
            'missing_texture': f"Missing Texture in: {issue.get('material', 'Unknown')}",
            'psd_file_detected': f"PSD File: {issue.get('texture', name)}",
            'large_texture': f"Large Texture ({issue.get('category', '')}): {issue.get('texture', name)}",
            'duplicate_materials': f"Duplicate Material: {issue.get('base_material', name)}",
            'duplicate_materials_identical': f"Identical Materials (Safe): {issue.get('base_material', name)}",
            'duplicate_materials_conflict': f"⚠ Naming Conflict: {issue.get('base_material', name)}",
            'duplicate_images': f"Duplicate Images: {name}",
            'zero_scale_object': f"Zero-Scale Object: {issue.get('object', name)}",
            'unconnected_texture_node': f"Unused Texture Node: {issue.get('material', 'Unknown')}",
            'unapplied_modifier': f"Unapplied Modifier: {issue.get('object', 'Unknown')}",
        }

        return titles.get(issue_type, f"{issue_type}: {name}")

    def draw(self, context):
        layout = self.layout
        scene = context.scene
        render_check = scene.cgmist_results

        # Marketing Header
        header_box = layout.box()
        header_col = header_box.column(align=True)
        header_col.scale_y = 1.2
        header_col.label(text="CGMist RenderCheck", icon='RENDERLAYERS')
        header_col.separator()
        
        # Marketing message
        msg_row = header_col.row()
        msg_row.alignment = 'CENTER'
        msg_row.label(text="Want to render in high quality and fast without overloading your system?")
        msg_row2 = header_col.row()
        msg_row2.alignment = 'CENTER'
        msg_row2.label(text="Render with us.")
        
        header_col.separator()
        row = header_col.row()
        row.scale_y = 1.3
        row.operator("cgmist.open_website", text="Visit CGMist.com", icon='URL')

        layout.separator()

        # Status Section
        status_box = layout.box()
        status_col = status_box.column(align=True)

        # Status header with icon
        if render_check.validation_status == "PASS":
            status_col.label(text="✓ Status: PASS - Ready to Render!", icon='CHECKMARK')
        elif render_check.validation_status == "WARNING":
            status_col.label(text="⚠ Status: WARNING - Review Recommended", icon='ERROR')
        elif render_check.validation_status == "NOT_RUN":
            status_col.label(text="Status: Not Yet Scanned", icon='QUESTION')
        else:
            status_col.label(text="✗ Status: FAIL - Issues Must Be Fixed", icon='CANCEL')

        if render_check.last_scan_time:
            status_col.label(text=f"Last scan: {render_check.last_scan_time}")

        # Issue Summary
        if render_check.validation_status != "NOT_RUN":
            summary_box = layout.box()
            summary_box.label(text="Issue Summary:", icon='INFO')
            
            summary_grid = summary_box.grid_flow(row_major=True, columns=4, even_columns=True, align=True)

            # Critical
            col = summary_grid.column()
            if render_check.critical_count > 0:
                col.alert = True
            col.label(text=f"Critical: {render_check.critical_count}")

            # High
            col = summary_grid.column()
            if render_check.high_count > 0:
                col.alert = True
            col.label(text=f"High: {render_check.high_count}")

            # Medium
            col = summary_grid.column()
            col.label(text=f"Medium: {render_check.medium_count}")

            # Low
            col = summary_grid.column()
            col.label(text=f"Low: {render_check.low_count}")

        # Issues List
        if render_check.total_count > 0:
            issues_box = layout.box()
            issues_box.label(text=f"Issues Found ({render_check.total_count}):", icon='ERROR')

            # Group issues by severity
            severity_order = ['CRITICAL', 'HIGH', 'MEDIUM', 'LOW']
            severity_icons = {
                'CRITICAL': 'CANCEL',
                'HIGH': 'ERROR',
                'MEDIUM': 'INFO',
                'LOW': 'DOT',
            }

            for severity in severity_order:
                severity_issues = [i for i in render_check.issues if i.severity == severity]
                if severity_issues:
                    sev_box = issues_box.box()
                    sev_box.label(text=f"{severity} ({len(severity_issues)})", icon=severity_icons[severity])

                    for idx, issue in enumerate(severity_issues):
                        row = sev_box.row(align=True)
                        row.label(text=f"  • {issue.title}")
                        
                        # Find actual index in the issues collection
                        actual_idx = list(render_check.issues).index(issue)
                        
                        if issue.can_auto_fix:
                            op = row.operator("cgmist.fix_single", text="Fix", icon='MODIFIER')
                            op.issue_index = actual_idx

            # Auto-Fix All button
            layout.separator()
            row = layout.row()
            row.scale_y = 1.5
            row.operator("cgmist.autofix_all", text="Auto-Fix All Safe Issues", icon='MODIFIER')

        elif render_check.validation_status == "PASS":
            pass_box = layout.box()
            pass_box.label(text="No issues found! Your file is ready for rendering.", icon='CHECKMARK')

        # Scene Info
        layout.separator()
        info_box = layout.box()
        info_box.label(text="Scene Information:", icon='SCENE_DATA')
        
        info_col = info_box.column(align=True)
        
        # File info
        filename = bpy.path.basename(bpy.data.filepath) if bpy.data.filepath else "Not Saved"
        info_col.label(text=f"File: {filename}")
        info_col.label(text=f"Engine: {scene.render.engine}")
        info_col.label(text=f"Frames: {scene.frame_start} - {scene.frame_end}")
        
        res_x = scene.render.resolution_x
        res_y = scene.render.resolution_y
        res_pct = scene.render.resolution_percentage
        eff_x = int(res_x * res_pct / 100)
        eff_y = int(res_y * res_pct / 100)
        info_col.label(text=f"Resolution: {eff_x}x{eff_y} ({res_pct}%)")
        info_col.label(text=f"Output: {scene.render.image_settings.file_format}")


# -----------------------------------------------------------------------------
# Registration
# -----------------------------------------------------------------------------

classes = [
    CGMIST_PG_Issue,
    CGMIST_PG_Results,
    CGMIST_OT_OpenWebsite,
    CGMIST_OT_Validate,
    CGMIST_OT_AutoFixAll,
    CGMIST_OT_FixSingle,
    CGMIST_OT_PreflightDialog,
]


def register():
    for cls in classes:
        bpy.utils.register_class(cls)

    # Register UI classes
    ui.register()

    # Add scene property
    bpy.types.Scene.cgmist_results = bpy.props.PointerProperty(type=CGMIST_PG_Results)


def unregister():
    # Remove scene property
    if hasattr(bpy.types.Scene, 'cgmist_results'):
        del bpy.types.Scene.cgmist_results

    # Unregister UI classes
    ui.unregister()

    for cls in reversed(classes):
        bpy.utils.unregister_class(cls)


if __name__ == "__main__":
    register()
