# SPDX-License-Identifier: GPL-3.0-or-later
"""
CGMist RenderCheck Fixer Module
Implements automatic fix functions for common render farm compatibility issues.
Includes smart deduplication for images and materials.

Compatible with Blender 3.0 - 5.0+
All functions are wrapped in try/except to prevent Blender crashes.
"""

import bpy
import os
import re


# Issue types that should NOT be auto-fixed (require manual review)
MANUAL_FIX_ONLY_ISSUES = {
    'zero_scale_object',           # Deleting objects is destructive - user must confirm
    'duplicate_materials_conflict', # Different materials with same base name - user must review
    'psd_file_detected',           # Requires external conversion
    'large_texture',               # Requires external resizing
    'missing_texture',             # User must provide the image
    'unconnected_texture_node',    # User must decide to connect or delete
    'unapplied_modifier',          # Applying is destructive
}


# -----------------------------------------------------------------------------
# Core Fix Functions
# -----------------------------------------------------------------------------

def fix_pack_resources():
    """
    Pack all external resources into the blend file.
    Uses bpy.ops.file.pack_all() to pack images, sounds, fonts, etc.

    Returns:
        tuple: (success: bool, message: str)
    """
    try:
        # Store current state for reporting
        unpacked_before = 0

        # Count unpacked images before
        for img in bpy.data.images:
            if img.type not in {'RENDER_RESULT', 'COMPOSITING'}:
                if img.filepath and not img.packed_file:
                    unpacked_before += 1

        # Count unpacked sounds
        for sound in bpy.data.sounds:
            if sound.filepath and not sound.packed_file:
                unpacked_before += 1

        # Count unpacked fonts
        for font in bpy.data.fonts:
            if font.filepath and not font.packed_file:
                unpacked_before += 1

        # Pack all resources
        bpy.ops.file.pack_all()

        # Count unpacked after
        unpacked_after = 0
        for img in bpy.data.images:
            if img.type not in {'RENDER_RESULT', 'COMPOSITING'}:
                if img.filepath and not img.packed_file:
                    unpacked_after += 1

        for sound in bpy.data.sounds:
            if sound.filepath and not sound.packed_file:
                unpacked_after += 1

        for font in bpy.data.fonts:
            if font.filepath and not font.packed_file:
                unpacked_after += 1

        packed_count = unpacked_before - unpacked_after

        if unpacked_after == 0:
            return True, f"Successfully packed {packed_count} resource(s)"
        else:
            return True, f"Packed {packed_count} resource(s), {unpacked_after} could not be packed (may be missing files)"

    except RuntimeError as e:
        return False, f"Pack operation failed: {str(e)}"
    except Exception as e:
        return False, f"Unexpected error during packing: {str(e)}"


def fix_relative_paths():
    """
    Convert all absolute file paths to relative paths (// prefix).
    This ensures paths work correctly when the blend file is moved.

    Returns:
        tuple: (success: bool, message: str)
    """
    try:
        converted_count = 0
        failed_count = 0

        # Only process if file is saved
        if not bpy.data.filepath:
            return False, "Please save the blend file first before converting to relative paths"

        blend_dir = os.path.dirname(bpy.data.filepath)

        # Convert image paths
        for img in bpy.data.images:
            if img.filepath and not img.filepath.startswith("//"):
                try:
                    # Check if file exists and is on same drive
                    abs_path = bpy.path.abspath(img.filepath)
                    if os.path.exists(abs_path):
                        img.filepath = bpy.path.relpath(abs_path)
                        converted_count += 1
                    else:
                        failed_count += 1
                except Exception:
                    failed_count += 1

        # Convert sound paths
        for sound in bpy.data.sounds:
            if sound.filepath and not sound.filepath.startswith("//"):
                try:
                    abs_path = bpy.path.abspath(sound.filepath)
                    if os.path.exists(abs_path):
                        sound.filepath = bpy.path.relpath(abs_path)
                        converted_count += 1
                    else:
                        failed_count += 1
                except Exception:
                    failed_count += 1

        # Convert font paths
        for font in bpy.data.fonts:
            if font.filepath and not font.filepath.startswith("//"):
                try:
                    abs_path = bpy.path.abspath(font.filepath)
                    if os.path.exists(abs_path):
                        font.filepath = bpy.path.relpath(abs_path)
                        converted_count += 1
                    else:
                        failed_count += 1
                except Exception:
                    failed_count += 1

        # Convert movie clip paths
        for clip in bpy.data.movieclips:
            if clip.filepath and not clip.filepath.startswith("//"):
                try:
                    abs_path = bpy.path.abspath(clip.filepath)
                    if os.path.exists(abs_path):
                        clip.filepath = bpy.path.relpath(abs_path)
                        converted_count += 1
                    else:
                        failed_count += 1
                except Exception:
                    failed_count += 1

        # Convert volume paths
        for volume in bpy.data.volumes:
            if volume.filepath and not volume.filepath.startswith("//"):
                try:
                    abs_path = bpy.path.abspath(volume.filepath)
                    if os.path.exists(abs_path):
                        volume.filepath = bpy.path.relpath(abs_path)
                        converted_count += 1
                    else:
                        failed_count += 1
                except Exception:
                    failed_count += 1

        if failed_count == 0:
            return True, f"Converted {converted_count} path(s) to relative"
        else:
            return True, f"Converted {converted_count} path(s), {failed_count} failed (files may be missing or on different drive)"

    except Exception as e:
        return False, f"Path conversion failed: {str(e)}"


def fix_render_settings():
    """
    Fix render output path if it uses an absolute path.

    STRICT RULES:
    - Do NOT change the user's Output Format (PNG, JPEG, EXR, etc.)
    - Only change the Output Path if it is absolute (not starting with //)
    - If already relative, leave it untouched

    Returns:
        tuple: (success: bool, message: str)
    """
    try:
        scene = bpy.context.scene
        changes = []

        # Check if output path is absolute (not relative)
        output_path = scene.render.filepath

        # A relative path in Blender starts with "//"
        # An absolute path does NOT start with "//" (e.g., "C:/Users/..." or "/home/user/...")
        if output_path and not output_path.startswith("//"):
            # Path is absolute - convert to relative
            scene.render.filepath = "//render/"
            changes.append(f"output path '{output_path}' -> //render/")

        if changes:
            return True, f"Applied: {', '.join(changes)}"
        else:
            return True, "Render settings already correct (output path is relative)"

    except Exception as e:
        return False, f"Render settings update failed: {str(e)}"


# -----------------------------------------------------------------------------
# Smart Image Deduplication
# -----------------------------------------------------------------------------

def fix_duplicate_images():
    """
    Smart Image Deduplication:
    Remap all users of duplicate image data blocks to the original,
    then delete the duplicate data blocks.
    
    This only merges images that point to the SAME file path.

    Returns:
        tuple: (success: bool, message: str)
    """
    try:
        # Group images by their absolute filepath
        filepath_to_images = {}
        
        for img in bpy.data.images:
            # Skip generated/render result images
            if img.type in {'RENDER_RESULT', 'COMPOSITING'}:
                continue
            
            if not img.filepath:
                continue
            
            # Get absolute path for comparison
            try:
                abs_path = bpy.path.abspath(img.filepath)
                abs_path = os.path.normpath(abs_path).lower()  # Normalize for comparison
            except:
                abs_path = img.filepath.lower()
            
            if abs_path not in filepath_to_images:
                filepath_to_images[abs_path] = []
            filepath_to_images[abs_path].append(img)
        
        total_remapped = 0
        total_removed = 0
        
        # Process each group with duplicates
        for abs_path, images in filepath_to_images.items():
            if len(images) <= 1:
                continue
            
            # Sort to determine the "original" (shortest name, no numeric suffix)
            images.sort(key=lambda x: (len(x.name), x.name))
            original = images[0]
            duplicates = images[1:]
            
            # Remap all users of duplicates to the original
            for dup in duplicates:
                try:
                    # Use Blender's user_remap to safely redirect all users
                    dup.user_remap(original)
                    total_remapped += 1
                except Exception:
                    pass
            
            # Remove duplicates with zero users
            for dup in duplicates:
                try:
                    if dup.users == 0:
                        bpy.data.images.remove(dup)
                        total_removed += 1
                except Exception:
                    pass
        
        if total_remapped > 0:
            return True, f"Remapped {total_remapped} duplicate image(s), removed {total_removed}"
        else:
            return True, "No duplicate images found"
        
    except Exception as e:
        return False, f"Image deduplication failed: {str(e)}"


def _fix_duplicate_images_specific(original_name, duplicate_names):
    """
    Fix specific duplicate images by remapping them to the original.
    
    Args:
        original_name: Name of the original image to keep
        duplicate_names: List of duplicate image names to remap and remove
    
    Returns:
        tuple: (success: bool, message: str)
    """
    try:
        if original_name not in bpy.data.images:
            return False, f"Original image '{original_name}' not found"
        
        original = bpy.data.images[original_name]
        remapped_count = 0
        removed_count = 0
        
        for dup_name in duplicate_names:
            if dup_name not in bpy.data.images:
                continue
            
            dup = bpy.data.images[dup_name]
            
            try:
                # Remap all users to the original
                dup.user_remap(original)
                remapped_count += 1
                
                # Remove if no users
                if dup.users == 0:
                    bpy.data.images.remove(dup)
                    removed_count += 1
            except Exception:
                pass
        
        return True, f"Remapped {remapped_count} duplicate(s) to '{original_name}', removed {removed_count}"
        
    except Exception as e:
        return False, f"Failed to deduplicate images: {str(e)}"


# -----------------------------------------------------------------------------
# Smart Material Deduplication
# -----------------------------------------------------------------------------

def fix_identical_materials():
    """
    Smart Material Deduplication:
    Only merges materials that are IDENTICAL in content.
    Materials with naming conflicts (different content) are NOT merged.

    Returns:
        tuple: (success: bool, message: str)
    """
    try:
        from . import validator
        
        # Get the current duplicate materials analysis
        duplicate_materials = validator.check_duplicate_materials()
        
        total_replaced = 0
        total_removed = 0
        
        for dup_info in duplicate_materials:
            # ONLY process identical materials
            if not dup_info.get('is_identical', False):
                continue
            
            base_name = dup_info.get('base_material', '')
            duplicates = dup_info.get('duplicates', [])
            
            if not base_name or base_name not in bpy.data.materials:
                continue
            
            base_mat = bpy.data.materials[base_name]
            
            # Replace duplicates on all objects
            for dup_name in duplicates:
                if dup_name not in bpy.data.materials:
                    continue
                
                dup_mat = bpy.data.materials[dup_name]
                
                # Use user_remap for safe remapping
                try:
                    dup_mat.user_remap(base_mat)
                    total_replaced += 1
                except:
                    # Fallback: manual remapping on objects
                    for obj in bpy.data.objects:
                        if not hasattr(obj, 'material_slots'):
                            continue
                        
                        for slot in obj.material_slots:
                            if slot.material and slot.material.name == dup_name:
                                slot.material = base_mat
                                total_replaced += 1
            
            # Remove unused duplicates
            for dup_name in duplicates:
                if dup_name in bpy.data.materials:
                    dup_mat = bpy.data.materials[dup_name]
                    if dup_mat.users == 0:
                        bpy.data.materials.remove(dup_mat)
                        total_removed += 1
        
        if total_replaced > 0:
            return True, f"Replaced {total_replaced} material reference(s), removed {total_removed} duplicate(s)"
        else:
            return True, "No identical duplicate materials found"
        
    except Exception as e:
        return False, f"Material deduplication failed: {str(e)}"


def _fix_identical_materials_specific(base_name, duplicate_names):
    """
    Fix specific identical materials by remapping them to the base material.
    
    Args:
        base_name: Name of the base material to keep
        duplicate_names: List of duplicate material names to remap and remove
    
    Returns:
        tuple: (success: bool, message: str)
    """
    try:
        if base_name not in bpy.data.materials:
            return False, f"Base material '{base_name}' not found"
        
        base_mat = bpy.data.materials[base_name]
        replaced_count = 0
        removed_count = 0
        
        for dup_name in duplicate_names:
            if dup_name not in bpy.data.materials:
                continue
            
            dup_mat = bpy.data.materials[dup_name]
            
            try:
                # Use user_remap for safe remapping
                dup_mat.user_remap(base_mat)
                replaced_count += 1
                
                # Remove if no users
                if dup_mat.users == 0:
                    bpy.data.materials.remove(dup_mat)
                    removed_count += 1
            except Exception:
                # Fallback: manual remapping
                for obj in bpy.data.objects:
                    if not hasattr(obj, 'material_slots'):
                        continue
                    
                    for slot in obj.material_slots:
                        if slot.material and slot.material.name == dup_name:
                            slot.material = base_mat
                            replaced_count += 1
        
        return True, f"Replaced {replaced_count} reference(s) with '{base_name}', removed {removed_count}"
        
    except Exception as e:
        return False, f"Failed to deduplicate materials: {str(e)}"


# -----------------------------------------------------------------------------
# Object Deletion (Manual Fix)
# -----------------------------------------------------------------------------

def fix_delete_object(obj_name):
    """
    Safely delete an object from the scene.

    This is used for removing zero-scale "ghost" objects that serve no purpose.

    Args:
        obj_name: The name of the object to delete

    Returns:
        tuple: (success: bool, message: str)
    """
    try:
        # Check if object exists
        if obj_name not in bpy.data.objects:
            return False, f"Object '{obj_name}' not found"

        obj = bpy.data.objects[obj_name]

        # Safety check: Don't delete objects with children
        if len(obj.children) > 0:
            return False, f"Object '{obj_name}' has children and cannot be safely deleted"

        # Safety check: Don't delete cameras or lights
        if obj.type in {'CAMERA', 'LIGHT'}:
            return False, f"Object '{obj_name}' is a {obj.type} and should not be deleted"

        # Store mesh data reference for cleanup (if it's a mesh)
        mesh_data = None
        if obj.type == 'MESH' and obj.data:
            mesh_data = obj.data

        # Remove object from all collections
        for collection in obj.users_collection:
            collection.objects.unlink(obj)

        # Remove from bpy.data.objects
        bpy.data.objects.remove(obj, do_unlink=True)

        # Clean up orphaned mesh data if no other objects use it
        if mesh_data and mesh_data.users == 0:
            bpy.data.meshes.remove(mesh_data)

        return True, f"Deleted object '{obj_name}'"

    except RuntimeError as e:
        return False, f"Could not delete '{obj_name}': {str(e)}"
    except Exception as e:
        return False, f"Delete failed for '{obj_name}': {str(e)}"


# -----------------------------------------------------------------------------
# Single Issue Fix Router
# -----------------------------------------------------------------------------

def fix_single_issue(issue_type, data_name, data_type, extra_data=None):
    """
    Attempt to fix a single specific issue.

    Args:
        issue_type: The type of issue (e.g., 'unpacked_image')
        data_name: The name of the affected data block
        data_type: The type of data (e.g., 'Image', 'Sound')
        extra_data: Optional dict with additional data for smart fixes

    Returns:
        tuple: (success: bool, message: str)
    """
    try:
        # Handle unpacked resources
        if issue_type.startswith('unpacked_'):
            return _fix_unpacked_item(issue_type, data_name, data_type)

        # Handle zero-scale objects (manual delete)
        if issue_type == 'zero_scale_object':
            return fix_delete_object(data_name)

        # Handle missing textures (can't auto-fix, provide guidance)
        if issue_type == 'missing_texture':
            return False, "Cannot auto-fix: Please assign an image to the texture node or delete it"

        # Handle PSD files (can't auto-fix)
        if issue_type == 'psd_file_detected':
            return False, "Cannot auto-fix: Please convert PSD to PNG/EXR manually and re-import"

        # Handle large textures (could downscale but that's destructive)
        if issue_type == 'large_texture':
            return False, "Cannot auto-fix: Please resize textures in external image editor"

        # Handle duplicate images (smart deduplication)
        if issue_type == 'duplicate_images':
            if extra_data:
                original = extra_data.get('original', '')
                duplicates = extra_data.get('duplicates', [])
                if original and duplicates:
                    return _fix_duplicate_images_specific(original, duplicates)
            return fix_duplicate_images()

        # Handle identical materials (safe to merge)
        if issue_type == 'duplicate_materials_identical':
            if extra_data:
                base_mat = extra_data.get('base_material', '')
                duplicates = extra_data.get('duplicates', [])
                if base_mat and duplicates:
                    return _fix_identical_materials_specific(base_mat, duplicates)
            return fix_identical_materials()

        # Handle material naming conflicts (DO NOT auto-merge)
        if issue_type == 'duplicate_materials_conflict':
            return False, "Cannot auto-fix: Materials have DIFFERENT content. Please review and rename or manually merge."

        # Handle legacy duplicate materials issue (from old validator)
        if issue_type == 'duplicate_materials':
            return False, "Please use the smart deduplication feature which checks material content before merging."

        # Handle unconnected nodes
        if issue_type == 'unconnected_texture_node':
            return False, "Cannot auto-fix: Please review and connect or delete unused nodes manually"

        # Handle unapplied modifiers
        if issue_type == 'unapplied_modifier':
            return False, "Cannot auto-fix: Applying modifiers is destructive - please do this manually"

        return False, f"Unknown issue type: {issue_type}"

    except Exception as e:
        return False, f"Fix failed: {str(e)}"


def _fix_unpacked_item(issue_type, data_name, data_type):
    """
    Pack a specific unpacked item.

    Returns:
        tuple: (success: bool, message: str)
    """
    try:
        if data_type == "Image" or issue_type == "unpacked_image":
            if data_name in bpy.data.images:
                img = bpy.data.images[data_name]
                img.pack()
                return True, f"Packed image: {data_name}"
            return False, f"Image not found: {data_name}"

        elif data_type == "Sound" or issue_type == "unpacked_sound":
            if data_name in bpy.data.sounds:
                sound = bpy.data.sounds[data_name]
                sound.pack()
                return True, f"Packed sound: {data_name}"
            return False, f"Sound not found: {data_name}"

        elif data_type == "Font" or issue_type == "unpacked_font":
            if data_name in bpy.data.fonts:
                font = bpy.data.fonts[data_name]
                font.pack()
                return True, f"Packed font: {data_name}"
            return False, f"Font not found: {data_name}"

        elif data_type == "Movie Clip" or issue_type == "unpacked_movie_clip":
            # Movie clips don't have a direct pack method, use operator
            return False, "Movie clips must be packed via File > External Data > Pack Resources"

        elif data_type == "Volume" or issue_type == "unpacked_volume":
            if data_name in bpy.data.volumes:
                volume = bpy.data.volumes[data_name]
                volume.pack()
                return True, f"Packed volume: {data_name}"
            return False, f"Volume not found: {data_name}"

        elif data_type == "Library" or issue_type == "unpacked_library":
            return False, "Libraries must be packed via File > External Data > Pack Resources"

        return False, f"Unknown data type: {data_type}"

    except RuntimeError as e:
        # Common error when file doesn't exist
        return False, f"Could not pack {data_name}: {str(e)}"
    except Exception as e:
        return False, f"Pack failed for {data_name}: {str(e)}"


# -----------------------------------------------------------------------------
# Utility Functions
# -----------------------------------------------------------------------------

def is_manual_fix_only(issue_type):
    """
    Check if an issue type requires manual fixing (excluded from Auto-Fix All).

    Args:
        issue_type: The type of issue to check

    Returns:
        bool: True if the issue should only be fixed manually
    """
    return issue_type in MANUAL_FIX_ONLY_ISSUES


def fix_all():
    """
    Run all automatic fixes in sequence.

    NOTE: This excludes issues marked as MANUAL_FIX_ONLY_ISSUES which
    require user confirmation or external tools before fixing.

    Returns:
        dict: Results of all fix operations
    """
    results = {}

    # Fix 1: Pack resources
    success, message = fix_pack_resources()
    results['pack_resources'] = {'success': success, 'message': message}

    # Fix 2: Convert to relative paths
    success, message = fix_relative_paths()
    results['relative_paths'] = {'success': success, 'message': message}

    # Fix 3: Fix render output path (if absolute)
    success, message = fix_render_settings()
    results['render_settings'] = {'success': success, 'message': message}

    # Fix 4: Smart image deduplication
    success, message = fix_duplicate_images()
    results['duplicate_images'] = {'success': success, 'message': message}

    # Fix 5: Smart material deduplication (identical only)
    success, message = fix_identical_materials()
    results['identical_materials'] = {'success': success, 'message': message}

    return results
