# SPDX-License-Identifier: GPL-3.0-or-later
"""
CGMist RenderCheck UI Module
Implements the N-Panel (Sidebar) interface and popup dialog for the CGMist RenderCheck addon.

Compatible with Blender 3.0 - 5.0+
"""

import bpy
from bpy.types import Panel, UIList

from . import fixer


# -----------------------------------------------------------------------------
# UI List for displaying issues
# -----------------------------------------------------------------------------

class CGMIST_UL_IssueList(UIList):
    """UI List for displaying validation issues."""

    def draw_item(self, context, layout, data, item, icon, active_data, active_property, index):
        if self.layout_type in {'DEFAULT', 'COMPACT'}:
            # Choose icon based on severity
            severity_icons = {
                'CRITICAL': 'CANCEL',
                'HIGH': 'ERROR',
                'MEDIUM': 'INFO',
                'LOW': 'DOT',
            }
            icon = severity_icons.get(item.severity, 'DOT')

            row = layout.row(align=True)

            # Severity indicator with color
            severity_colors = {
                'CRITICAL': (1.0, 0.2, 0.2, 1.0),
                'HIGH': (1.0, 0.5, 0.2, 1.0),
                'MEDIUM': (1.0, 0.8, 0.2, 1.0),
                'LOW': (0.5, 0.7, 1.0, 1.0),
            }

            # Display severity badge and title
            row.label(text="", icon=icon)
            
            # Truncate title if too long
            title = item.title
            if len(title) > 40:
                title = title[:37] + "..."
            row.label(text=title)
            
            # Show auto-fix indicator
            if item.can_auto_fix:
                row.label(text="", icon='MODIFIER')

        elif self.layout_type == 'GRID':
            layout.alignment = 'CENTER'
            layout.label(text="", icon='MATERIAL')


# -----------------------------------------------------------------------------
# Main Panel
# -----------------------------------------------------------------------------

class CGMIST_PT_MainPanel(Panel):
    """Main panel for CGMist RenderCheck in the N-Panel sidebar."""
    bl_label = "CGMist RenderCheck"
    bl_idname = "CGMIST_PT_main"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "CGMist"

    def draw(self, context):
        layout = self.layout
        scene = context.scene
        render_check = scene.cgmist_results

        # Marketing Header Box
        header_box = layout.box()
        header_col = header_box.column(align=True)
        header_col.scale_y = 0.9
        
        # Marketing message (wrapped for narrow panel)
        header_col.label(text="Want to render fast")
        header_col.label(text="without overloading")
        header_col.label(text="your system?")
        header_col.separator()
        header_col.label(text="Render with us.", icon='HEART')
        
        header_col.separator()
        row = header_col.row()
        row.scale_y = 1.2
        row.operator("cgmist.open_website", text="Visit CGMist.com", icon='URL')

        layout.separator()

        # Main Pre-flight Button
        box = layout.box()
        col = box.column(align=True)
        col.scale_y = 1.8
        col.operator("cgmist.preflight_dialog", text="Start CGMist Pre-flight", icon='VIEWZOOM')
        
        col.separator()
        col.scale_y = 1.0
        col.label(text="Opens full validation report", icon='INFO')

        layout.separator()

        # Quick Scan button (stays in panel)
        row = layout.row(align=True)
        row.scale_y = 1.3
        row.operator("cgmist.validate", text="Quick Scan", icon='FILE_REFRESH')

        # Compact Status display
        if render_check.validation_status != "NOT_RUN":
            box = layout.box()

            # Status header with icon
            status_row = box.row()
            if render_check.validation_status == "PASS":
                status_row.label(text="✓ PASS", icon='CHECKMARK')
            elif render_check.validation_status == "WARNING":
                status_row.label(text="⚠ WARNING", icon='ERROR')
            else:
                status_row.label(text="✗ FAIL", icon='CANCEL')

            # Compact issue counts
            if render_check.total_count > 0:
                counts = []
                if render_check.critical_count > 0:
                    counts.append(f"C:{render_check.critical_count}")
                if render_check.high_count > 0:
                    counts.append(f"H:{render_check.high_count}")
                if render_check.medium_count > 0:
                    counts.append(f"M:{render_check.medium_count}")
                if render_check.low_count > 0:
                    counts.append(f"L:{render_check.low_count}")
                
                box.label(text=" | ".join(counts))
            else:
                box.label(text="No issues found!", icon='CHECKMARK')


class CGMIST_PT_IssuesPanel(Panel):
    """Panel for displaying the list of issues."""
    bl_label = "Issues"
    bl_idname = "CGMIST_PT_issues"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "CGMist"
    bl_parent_id = "CGMIST_PT_main"
    bl_options = {'DEFAULT_CLOSED'}

    @classmethod
    def poll(cls, context):
        render_check = context.scene.cgmist_results
        return render_check.validation_status != "NOT_RUN" and render_check.total_count > 0

    def draw(self, context):
        layout = self.layout
        scene = context.scene
        render_check = scene.cgmist_results

        # Issue list
        row = layout.row()
        row.template_list(
            "CGMIST_UL_IssueList", "",
            render_check, "issues",
            render_check, "active_issue_index",
            rows=5
        )

        # Selected issue details
        if render_check.issues and 0 <= render_check.active_issue_index < len(render_check.issues):
            issue = render_check.issues[render_check.active_issue_index]

            box = layout.box()
            box.label(text="Details:", icon='INFO')

            # Severity
            row = box.row()
            severity_icons = {
                'CRITICAL': 'CANCEL',
                'HIGH': 'ERROR',
                'MEDIUM': 'INFO',
                'LOW': 'DOT',
            }
            row.label(text=f"Severity: {issue.severity}", icon=severity_icons.get(issue.severity, 'DOT'))

            # Description (wrapped)
            if issue.description:
                col = box.column(align=True)
                col.scale_y = 0.8
                self._draw_wrapped_text(col, issue.description, 35)

            # Fix instructions
            if issue.fix_instructions:
                box.separator()
                box.label(text="How to Fix:", icon='TOOL_SETTINGS')
                col = box.column(align=True)
                col.scale_y = 0.8
                self._draw_wrapped_text(col, issue.fix_instructions, 35)

            # Single issue fix button
            box.separator()
            row = box.row()
            
            if issue.can_auto_fix:
                op = row.operator("cgmist.fix_single", text="Fix This Issue", icon='MODIFIER')
                op.issue_index = render_check.active_issue_index
            else:
                row.enabled = False
                row.operator("cgmist.fix_single", text="Manual Fix Required", icon='LOCKED')

            # Show warning for destructive operations
            if issue.issue_type == 'zero_scale_object':
                warn_box = box.box()
                warn_box.alert = True
                col = warn_box.column(align=True)
                col.scale_y = 0.8
                col.label(text="Warning: This will DELETE", icon='ERROR')
                col.label(text="the object!")

            # Show warning for naming conflicts
            if issue.issue_type == 'duplicate_materials_conflict':
                warn_box = box.box()
                warn_box.alert = True
                col = warn_box.column(align=True)
                col.scale_y = 0.8
                col.label(text="⚠ Materials are DIFFERENT!", icon='ERROR')
                col.label(text="Review before merging.")

    def _draw_wrapped_text(self, layout, text, width):
        """Draw text with word wrapping."""
        words = text.split()
        lines = []
        current_line = ""

        for word in words:
            if len(current_line) + len(word) + 1 <= width:
                current_line += (" " if current_line else "") + word
            else:
                if current_line:
                    lines.append(current_line)
                current_line = word

        if current_line:
            lines.append(current_line)

        for line in lines:
            layout.label(text=line)


class CGMIST_PT_FixPanel(Panel):
    """Panel for auto-fix controls."""
    bl_label = "Auto-Fix"
    bl_idname = "CGMIST_PT_fix"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "CGMist"
    bl_parent_id = "CGMIST_PT_main"
    bl_options = {'DEFAULT_CLOSED'}

    @classmethod
    def poll(cls, context):
        render_check = context.scene.cgmist_results
        return render_check.validation_status != "NOT_RUN"

    def draw(self, context):
        layout = self.layout
        render_check = context.scene.cgmist_results

        # Auto-fix all button
        col = layout.column(align=True)

        row = col.row(align=True)
        row.scale_y = 1.3
        row.operator("cgmist.autofix_all", text="Auto-Fix All", icon='MODIFIER')

        # Info text
        box = layout.box()
        box.label(text="Auto-Fix will:", icon='INFO')
        col = box.column(align=True)
        col.scale_y = 0.8
        col.label(text="• Pack external resources")
        col.label(text="• Convert to relative paths")
        col.label(text="• Fix absolute output paths")
        col.label(text="• Merge duplicate images")
        col.label(text="• Merge identical materials")

        # Warning about what it won't change
        box = layout.box()
        col = box.column(align=True)
        col.scale_y = 0.8
        col.label(text="Will NOT change:", icon='LOCKED')
        col.label(text="• Output format")
        col.label(text="• Relative output paths")
        col.label(text="• Different materials")
        col.label(text="  (naming conflicts)")

        # Warning about manual-only fixes
        box = layout.box()
        box.alert = True
        col = box.column(align=True)
        col.scale_y = 0.8
        col.label(text="Requires manual fix:", icon='ERROR')
        col.label(text="• Zero-scale objects")
        col.label(text="• PSD files")
        col.label(text="• Large textures")
        col.label(text="• Material conflicts")


class CGMIST_PT_SceneInfoPanel(Panel):
    """Panel for displaying scene information."""
    bl_label = "Scene Info"
    bl_idname = "CGMIST_PT_scene_info"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "CGMist"
    bl_parent_id = "CGMIST_PT_main"
    bl_options = {'DEFAULT_CLOSED'}

    def draw(self, context):
        layout = self.layout
        scene = context.scene

        col = layout.column(align=True)

        # File info
        filename = bpy.path.basename(bpy.data.filepath) if bpy.data.filepath else "Not Saved"
        col.label(text=f"File: {filename}")

        # Render engine
        col.label(text=f"Engine: {scene.render.engine}")

        # Frame range
        total_frames = scene.frame_end - scene.frame_start + 1
        col.label(text=f"Frames: {scene.frame_start}-{scene.frame_end} ({total_frames})")

        # Resolution
        res_x = scene.render.resolution_x
        res_y = scene.render.resolution_y
        res_pct = scene.render.resolution_percentage
        eff_x = int(res_x * res_pct / 100)
        eff_y = int(res_y * res_pct / 100)
        col.label(text=f"Res: {eff_x}x{eff_y} ({res_pct}%)")

        # Output format
        col.label(text=f"Output: {scene.render.image_settings.file_format}")
        
        # Output path (truncated)
        output_path = scene.render.filepath
        if len(output_path) > 25:
            output_path = "..." + output_path[-22:]
        col.label(text=f"Path: {output_path}")


class CGMIST_PT_SmartDeduplicationPanel(Panel):
    """Panel explaining smart deduplication features."""
    bl_label = "Smart Deduplication"
    bl_idname = "CGMIST_PT_smart_dedup"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "CGMist"
    bl_parent_id = "CGMIST_PT_main"
    bl_options = {'DEFAULT_CLOSED'}

    def draw(self, context):
        layout = self.layout

        # Image deduplication info
        box = layout.box()
        box.label(text="Image Deduplication", icon='IMAGE_DATA')
        col = box.column(align=True)
        col.scale_y = 0.8
        col.label(text="Detects images that point")
        col.label(text="to the same file but are")
        col.label(text="separate data blocks.")
        col.label(text="")
        col.label(text="Safe to auto-merge.")

        # Material deduplication info
        box = layout.box()
        box.label(text="Material Deduplication", icon='MATERIAL')
        col = box.column(align=True)
        col.scale_y = 0.8
        col.label(text="Compares material content,")
        col.label(text="not just names.")
        col.label(text="")
        col.label(text="• Identical: Safe to merge")
        col.label(text="• Different: Flagged as")
        col.label(text="  naming conflict")


# -----------------------------------------------------------------------------
# Registration
# -----------------------------------------------------------------------------

classes = [
    CGMIST_UL_IssueList,
    CGMIST_PT_MainPanel,
    CGMIST_PT_IssuesPanel,
    CGMIST_PT_FixPanel,
    CGMIST_PT_SceneInfoPanel,
    CGMIST_PT_SmartDeduplicationPanel,
]


def register():
    for cls in classes:
        bpy.utils.register_class(cls)


def unregister():
    for cls in reversed(classes):
        bpy.utils.unregister_class(cls)
