# SPDX-License-Identifier: GPL-3.0-or-later
"""
CGMist RenderCheck Validator Module
Contains all validation logic for checking blend file compatibility with render farms.
Includes smart deduplication detection for images and materials.

Compatible with Blender 3.0 - 5.0+
"""

import bpy
import os
import re
from datetime import datetime


# Threshold for considering a value "near zero"
ZERO_THRESHOLD = 0.0001


# -----------------------------------------------------------------------------
# Blender Version Compatibility
# -----------------------------------------------------------------------------

def get_blender_version():
    """Get Blender version as tuple."""
    return bpy.app.version


def is_blender_4_plus():
    """Check if Blender 4.0 or higher."""
    return bpy.app.version >= (4, 0, 0)


def get_eevee_engine_name():
    """Get the correct EEVEE engine name for the current Blender version."""
    if bpy.app.version >= (4, 2, 0):
        return 'BLENDER_EEVEE_NEXT'
    elif bpy.app.version >= (4, 0, 0):
        return 'BLENDER_EEVEE_NEXT'
    else:
        return 'BLENDER_EEVEE'


# -----------------------------------------------------------------------------
# Scene Information
# -----------------------------------------------------------------------------

def get_scene_info():
    """Get comprehensive scene information."""
    try:
        scene = bpy.context.scene

        info = {
            "filename": os.path.basename(bpy.data.filepath) if bpy.data.filepath else "untitled.blend",
            "render_engine": scene.render.engine,
            "frame_range": {
                "start": scene.frame_start,
                "end": scene.frame_end,
                "total": scene.frame_end - scene.frame_start + 1
            },
            "resolution": {
                "width": scene.render.resolution_x,
                "height": scene.render.resolution_y,
                "percentage": scene.render.resolution_percentage,
                "effective": f"{int(scene.render.resolution_x * scene.render.resolution_percentage / 100)}x{int(scene.render.resolution_y * scene.render.resolution_percentage / 100)}"
            },
            "blender_version": ".".join(map(str, bpy.app.version))
        }

        # Engine-specific settings with version compatibility
        if scene.render.engine == 'CYCLES':
            cycles_info = {
                "samples": scene.cycles.samples,
                "device": scene.cycles.device,
            }
            
            # Denoising settings vary by version
            if hasattr(scene.cycles, 'use_denoising'):
                cycles_info["use_denoising"] = scene.cycles.use_denoising
                if scene.cycles.use_denoising and hasattr(scene.cycles, 'denoiser'):
                    cycles_info["denoiser"] = scene.cycles.denoiser
            
            # Adaptive sampling
            if hasattr(scene.cycles, 'use_adaptive_sampling'):
                cycles_info["adaptive_sampling"] = scene.cycles.use_adaptive_sampling
            
            # Time limit
            if hasattr(scene.cycles, 'time_limit'):
                cycles_info["time_limit"] = scene.cycles.time_limit if scene.cycles.time_limit > 0 else None
                
            info["cycles"] = cycles_info
            
        elif scene.render.engine in ('BLENDER_EEVEE_NEXT', 'BLENDER_EEVEE'):
            eevee_info = {}
            
            # Sample settings vary by version
            if hasattr(scene.eevee, 'taa_render_samples'):
                eevee_info["samples"] = scene.eevee.taa_render_samples
            elif hasattr(scene.eevee, 'samples'):
                eevee_info["samples"] = scene.eevee.samples
            
            # Legacy EEVEE settings (Blender 3.x)
            if hasattr(scene.eevee, 'use_gtao'):
                eevee_info["use_gtao"] = scene.eevee.use_gtao
            if hasattr(scene.eevee, 'use_bloom'):
                eevee_info["use_bloom"] = scene.eevee.use_bloom
            if hasattr(scene.eevee, 'use_ssr'):
                eevee_info["use_ssr"] = scene.eevee.use_ssr
            if hasattr(scene.eevee, 'use_motion_blur'):
                eevee_info["use_motion_blur"] = scene.eevee.use_motion_blur
                
            info["eevee"] = eevee_info

        return info
    except Exception as e:
        return {"error": str(e)}


# -----------------------------------------------------------------------------
# Critical Checks
# -----------------------------------------------------------------------------

def check_packed_status():
    """Check if all external data is packed - CRITICAL."""
    unpacked_items = []

    try:
        # Check images
        for img in bpy.data.images:
            if img.type in {'RENDER_RESULT', 'COMPOSITING'}:
                continue

            if img.filepath and not img.packed_file:
                unpacked_items.append({
                    "issue": "unpacked_image",
                    "type": "Image",
                    "name": img.name,
                    "filepath": img.filepath,
                    "technical_detail": f"Image '{img.name}' references external file but packed_file is None",
                    "user_message": f"The texture '{img.name}' is not packed into your .blend file. The render farm won't have access to this file, causing materials to appear pink or missing.",
                    "fix_instructions": "Go to File > External Data > Pack Resources, then save your file. Or select the image in the Shader Editor and click the shield icon to pack it individually.",
                    "can_auto_fix": True
                })
    except Exception:
        pass

    try:
        # Check movie clips
        for clip in bpy.data.movieclips:
            if clip.filepath and not clip.packed_file:
                unpacked_items.append({
                    "issue": "unpacked_movie_clip",
                    "type": "Movie Clip",
                    "name": clip.name,
                    "filepath": clip.filepath,
                    "technical_detail": f"Movie clip '{clip.name}' references external file but packed_file is None",
                    "user_message": f"The movie clip '{clip.name}' is not packed. Any motion tracking data or video textures will be missing during rendering.",
                    "fix_instructions": "Go to File > External Data > Pack Resources to pack all external files, then save.",
                    "can_auto_fix": True
                })
    except Exception:
        pass

    try:
        # Check sounds
        for sound in bpy.data.sounds:
            if sound.filepath and not sound.packed_file:
                unpacked_items.append({
                    "issue": "unpacked_sound",
                    "type": "Sound",
                    "name": sound.name,
                    "filepath": sound.filepath,
                    "technical_detail": f"Sound '{sound.name}' references external file but packed_file is None",
                    "user_message": f"The audio file '{sound.name}' is not packed. Your rendered animation will have no sound.",
                    "fix_instructions": "Go to File > External Data > Pack Resources to pack all external files, then save.",
                    "can_auto_fix": True
                })
    except Exception:
        pass

    try:
        # Check volumes
        for volume in bpy.data.volumes:
            if volume.filepath and not volume.packed_file:
                unpacked_items.append({
                    "issue": "unpacked_volume",
                    "type": "Volume",
                    "name": volume.name,
                    "filepath": volume.filepath,
                    "technical_detail": f"Volume '{volume.name}' references external file but packed_file is None",
                    "user_message": f"The volume data '{volume.name}' (smoke, fire, VDB) is not packed. Your volume effects will be missing from renders.",
                    "fix_instructions": "Go to File > External Data > Pack Resources to pack all external files, then save.",
                    "can_auto_fix": True
                })
    except Exception:
        pass

    try:
        # Check fonts
        for font in bpy.data.fonts:
            if font.filepath and not font.packed_file:
                unpacked_items.append({
                    "issue": "unpacked_font",
                    "type": "Font",
                    "name": font.name,
                    "filepath": font.filepath,
                    "technical_detail": f"Font '{font.name}' references external file but packed_file is None",
                    "user_message": f"The font '{font.name}' is not packed. Text objects using this font will render incorrectly or with a default font.",
                    "fix_instructions": "Go to File > External Data > Pack Resources to pack all external files, then save.",
                    "can_auto_fix": True
                })
    except Exception:
        pass

    try:
        # Check libraries (linked data)
        for lib in bpy.data.libraries:
            if lib.filepath and not lib.packed_file:
                unpacked_items.append({
                    "issue": "unpacked_library",
                    "type": "Library",
                    "name": lib.name,
                    "filepath": lib.filepath,
                    "technical_detail": f"Library '{lib.name}' references external .blend file but packed_file is None",
                    "user_message": f"The linked library '{lib.name}' is not packed. Any linked objects, materials, or assets from this library will be missing.",
                    "fix_instructions": "Go to File > External Data > Pack Resources to pack linked libraries, then save. Alternatively, append the data instead of linking it.",
                    "can_auto_fix": True
                })
    except Exception:
        pass

    return unpacked_items


def check_missing_textures():
    """Check for Image Texture nodes with no image assigned - CRITICAL."""
    missing_textures = []

    try:
        # Check all materials
        for mat in bpy.data.materials:
            if not mat.use_nodes or not mat.node_tree:
                continue

            try:
                for node in mat.node_tree.nodes:
                    if node.type == 'TEX_IMAGE' and not node.image:
                        missing_textures.append({
                            "issue": "missing_texture",
                            "material": mat.name,
                            "name": mat.name,
                            "node": node.name if node.name else "Image Texture",
                            "technical_detail": f"Image Texture node in material '{mat.name}' has node.image = None",
                            "user_message": f"The material '{mat.name}' has an Image Texture node with no image assigned. This will cause rendering errors or unexpected material appearance.",
                            "fix_instructions": f"Open the Shader Editor, select material '{mat.name}', find the Image Texture node, and either assign an image or delete the unused node.",
                            "can_auto_fix": False
                        })
            except Exception:
                continue
    except Exception:
        pass

    try:
        # Check world shader nodes
        for world in bpy.data.worlds:
            if not world.use_nodes or not world.node_tree:
                continue

            try:
                for node in world.node_tree.nodes:
                    if node.type == 'TEX_IMAGE' and not node.image:
                        missing_textures.append({
                            "issue": "missing_texture",
                            "material": f"World-{world.name}",
                            "name": f"World-{world.name}",
                            "node": node.name if node.name else "Image Texture",
                            "technical_detail": f"Image Texture node in world '{world.name}' has node.image = None",
                            "user_message": f"The world shader '{world.name}' has an Image Texture node with no image assigned. Your environment/background may render incorrectly.",
                            "fix_instructions": f"Open the Shader Editor in World mode, find the Image Texture node, and either assign an image or delete the unused node.",
                            "can_auto_fix": False
                        })
            except Exception:
                continue
    except Exception:
        pass

    return missing_textures


def check_psd_files():
    """Check if any Image Texture nodes use .psd files - CRITICAL."""
    psd_files = []

    try:
        # Check all materials
        for mat in bpy.data.materials:
            if not mat.use_nodes or not mat.node_tree:
                continue

            try:
                for node in mat.node_tree.nodes:
                    if node.type == 'TEX_IMAGE' and node.image:
                        img = node.image
                        if img.filepath and img.filepath.lower().endswith('.psd'):
                            psd_files.append({
                                "issue": "psd_file_detected",
                                "material": mat.name,
                                "texture": img.name,
                                "name": img.name,
                                "filepath": img.filepath,
                                "technical_detail": f"Image '{img.name}' uses .psd format which Blender cannot pack properly",
                                "user_message": f"The texture '{img.name}' is a .psd file. Blender does not pack PSD files correctly, so this texture will be missing on the render farm, causing pink/blank materials.",
                                "fix_instructions": "Open the PSD in Photoshop/GIMP and save it as PNG (for color/transparency) or EXR (for HDR). Then replace the texture in Blender's Shader Editor and pack the file.",
                                "can_auto_fix": False
                            })
            except Exception:
                continue
    except Exception:
        pass

    try:
        # Check world shader nodes
        for world in bpy.data.worlds:
            if not world.use_nodes or not world.node_tree:
                continue

            try:
                for node in world.node_tree.nodes:
                    if node.type == 'TEX_IMAGE' and node.image:
                        img = node.image
                        if img.filepath and img.filepath.lower().endswith('.psd'):
                            psd_files.append({
                                "issue": "psd_file_detected",
                                "material": f"World-{world.name}",
                                "texture": img.name,
                                "name": img.name,
                                "filepath": img.filepath,
                                "technical_detail": f"Image '{img.name}' uses .psd format which Blender cannot pack properly",
                                "user_message": f"The world texture '{img.name}' is a .psd file. Blender does not pack PSD files correctly, so your environment/background will be missing during rendering.",
                                "fix_instructions": "Convert the PSD to PNG or EXR format, then replace it in the world shader and pack the file.",
                                "can_auto_fix": False
                            })
            except Exception:
                continue
    except Exception:
        pass

    return psd_files


# -----------------------------------------------------------------------------
# Smart Deduplication Checks
# -----------------------------------------------------------------------------

def check_duplicate_images():
    """
    Check for duplicate image data blocks pointing to the same file - HIGH.
    
    Smart Image Deduplication:
    Detects images that are separate data blocks (e.g., "Texture.png" and "Texture.001.png")
    but point to the same absolute file path.
    """
    duplicates = []
    
    try:
        # Group images by their absolute filepath
        filepath_to_images = {}
        
        for img in bpy.data.images:
            # Skip generated/render result images
            if img.type in {'RENDER_RESULT', 'COMPOSITING'}:
                continue
            
            if not img.filepath:
                continue
            
            # Get absolute path for comparison
            try:
                abs_path = bpy.path.abspath(img.filepath)
                abs_path = os.path.normpath(abs_path).lower()  # Normalize for comparison
            except:
                abs_path = img.filepath.lower()
            
            if abs_path not in filepath_to_images:
                filepath_to_images[abs_path] = []
            filepath_to_images[abs_path].append(img)
        
        # Find groups with more than one image
        for abs_path, images in filepath_to_images.items():
            if len(images) > 1:
                # Sort by name to determine the "original" (shortest name, no suffix)
                images.sort(key=lambda x: (len(x.name), x.name))
                original = images[0]
                dupes = images[1:]
                
                dupe_names = [img.name for img in dupes]
                
                duplicates.append({
                    "issue": "duplicate_images",
                    "name": original.name,
                    "original_image": original.name,
                    "duplicate_images": dupe_names,
                    "filepath": abs_path,
                    "count": len(dupes),
                    "technical_detail": f"Found {len(dupes)} duplicate image data blocks pointing to '{os.path.basename(abs_path)}'",
                    "user_message": f"Found {len(dupes) + 1} image data blocks ({original.name}, {', '.join(dupe_names)}) all pointing to the same file. This wastes memory as the image is loaded multiple times.",
                    "fix_instructions": f"The duplicates will be remapped to use '{original.name}' and then removed. This is safe as they all reference the same file.",
                    "can_auto_fix": True,
                    "extra_data": {
                        "original": original.name,
                        "duplicates": dupe_names
                    }
                })
    except Exception:
        pass
    
    return duplicates


def _get_material_signature(mat):
    """
    Generate a signature for a material based on its node tree content.
    
    This compares:
    - Node types and their parameters
    - Node connections
    - Texture references
    
    Returns a hashable signature for comparison.
    """
    if not mat.use_nodes or not mat.node_tree:
        return None
    
    try:
        signature_parts = []
        node_tree = mat.node_tree
        
        # Sort nodes by type and name for consistent comparison
        nodes = sorted(node_tree.nodes, key=lambda n: (n.type, n.name))
        
        for node in nodes:
            node_sig = [node.type]
            
            # Add relevant node properties based on type
            if node.type == 'BSDF_PRINCIPLED':
                # Check key inputs
                for input_name in ['Base Color', 'Metallic', 'Roughness', 'IOR', 'Alpha']:
                    inp = node.inputs.get(input_name)
                    if inp and not inp.is_linked:
                        if hasattr(inp, 'default_value'):
                            val = inp.default_value
                            if hasattr(val, '__iter__'):
                                node_sig.append(f"{input_name}:{tuple(round(v, 4) for v in val)}")
                            else:
                                node_sig.append(f"{input_name}:{round(val, 4)}")
                                
            elif node.type == 'TEX_IMAGE':
                # Include image reference
                if node.image:
                    # Use the absolute filepath for comparison
                    try:
                        img_path = bpy.path.abspath(node.image.filepath) if node.image.filepath else node.image.name
                        node_sig.append(f"image:{os.path.normpath(img_path).lower()}")
                    except:
                        node_sig.append(f"image:{node.image.name}")
                else:
                    node_sig.append("image:None")
                    
                # Include projection and extension settings
                if hasattr(node, 'projection'):
                    node_sig.append(f"proj:{node.projection}")
                if hasattr(node, 'extension'):
                    node_sig.append(f"ext:{node.extension}")
                    
            elif node.type == 'MIX_RGB' or node.type == 'MIX':
                if hasattr(node, 'blend_type'):
                    node_sig.append(f"blend:{node.blend_type}")
                    
            elif node.type == 'MATH':
                if hasattr(node, 'operation'):
                    node_sig.append(f"op:{node.operation}")
                    
            elif node.type in ('RGB', 'VALUE'):
                # Color or value nodes
                for output in node.outputs:
                    if hasattr(output, 'default_value'):
                        val = output.default_value
                        if hasattr(val, '__iter__'):
                            node_sig.append(f"val:{tuple(round(v, 4) for v in val)}")
                        else:
                            node_sig.append(f"val:{round(val, 4)}")
            
            signature_parts.append(tuple(node_sig))
        
        # Add connection information
        connections = []
        for link in node_tree.links:
            conn = (
                link.from_node.type if link.from_node else "None",
                link.from_socket.name if link.from_socket else "None",
                link.to_node.type if link.to_node else "None",
                link.to_socket.name if link.to_socket else "None"
            )
            connections.append(conn)
        
        connections.sort()
        signature_parts.append(tuple(connections))
        
        return tuple(signature_parts)
        
    except Exception:
        return None


def check_duplicate_materials():
    """
    Check for duplicate materials with SMART comparison - MEDIUM/HIGH.
    
    Smart Material Deduplication:
    Before flagging materials like "Material.001" for merging into "Material",
    we compare their actual content (nodes, parameters, textures).
    
    - If Identical: Flag as safe to auto-merge (MEDIUM)
    - If Different: Flag as naming conflict, DO NOT auto-merge (HIGH)
    """
    duplicates = []
    
    try:
        pattern = re.compile(r'^(.+)\.(\d{3})$')
        
        # Group materials by base name
        material_groups = {}
        
        for mat in bpy.data.materials:
            match = pattern.match(mat.name)
            if match:
                base_name = match.group(1)
                if base_name not in material_groups:
                    material_groups[base_name] = {'base': None, 'variants': []}
                material_groups[base_name]['variants'].append(mat)
            else:
                # This could be a base material
                if mat.name not in material_groups:
                    material_groups[mat.name] = {'base': None, 'variants': []}
                material_groups[mat.name]['base'] = mat
        
        # Analyze each group
        for base_name, group in material_groups.items():
            if not group['variants']:
                continue
            
            base_mat = group['base']
            variants = group['variants']
            
            if not base_mat:
                # No base material exists, can't merge
                continue
            
            # Get signature for base material
            base_sig = _get_material_signature(base_mat)
            
            identical_variants = []
            different_variants = []
            
            for var_mat in variants:
                var_sig = _get_material_signature(var_mat)
                
                if base_sig is not None and var_sig is not None and base_sig == var_sig:
                    identical_variants.append(var_mat.name)
                else:
                    different_variants.append(var_mat.name)
            
            # Report identical materials (safe to merge)
            if identical_variants:
                duplicates.append({
                    "issue": "duplicate_materials_identical",
                    "base_material": base_name,
                    "name": base_name,
                    "duplicates": identical_variants,
                    "count": len(identical_variants),
                    "is_identical": True,
                    "technical_detail": f"Found {len(identical_variants)} identical material(s) with base name '{base_name}'",
                    "user_message": f"Found {len(identical_variants)} duplicate(s) of material '{base_name}' ({', '.join(identical_variants)}) that are IDENTICAL. These can be safely merged.",
                    "fix_instructions": f"Auto-fix will remap all objects using these duplicates to '{base_name}' and remove the unused copies.",
                    "can_auto_fix": True,
                    "extra_data": {
                        "base_material": base_name,
                        "duplicates": identical_variants,
                        "is_identical": True
                    }
                })
            
            # Report different materials (naming conflict - DO NOT auto-merge)
            if different_variants:
                duplicates.append({
                    "issue": "duplicate_materials_conflict",
                    "base_material": base_name,
                    "name": base_name,
                    "duplicates": different_variants,
                    "count": len(different_variants),
                    "is_identical": False,
                    "technical_detail": f"Found {len(different_variants)} material(s) with base name '{base_name}' that have DIFFERENT content",
                    "user_message": f"⚠ NAMING CONFLICT: Materials '{', '.join(different_variants)}' have the same base name as '{base_name}' but are DIFFERENT materials! Auto-merge is disabled to prevent data loss.",
                    "fix_instructions": f"Review these materials in the Shader Editor. If they should be different, rename them to unique names. If they should be the same, manually verify and merge.",
                    "can_auto_fix": False,
                    "extra_data": {
                        "base_material": base_name,
                        "duplicates": different_variants,
                        "is_identical": False
                    }
                })
    except Exception:
        pass
    
    return duplicates


# -----------------------------------------------------------------------------
# Medium/Low Priority Checks
# -----------------------------------------------------------------------------

def check_large_textures():
    """Check for large textures (4K or 8K) - MEDIUM."""
    large_textures = []
    processed_images = set()

    try:
        # Check all materials
        for mat in bpy.data.materials:
            if not mat.use_nodes or not mat.node_tree:
                continue

            try:
                for node in mat.node_tree.nodes:
                    if node.type == 'TEX_IMAGE' and node.image:
                        img = node.image
                        img_id = img.name

                        if img_id in processed_images:
                            continue

                        width = img.size[0]
                        height = img.size[1]

                        category = None
                        if width >= 8192 or height >= 8192:
                            category = "8K"
                        elif width >= 4096 or height >= 4096:
                            category = "4K"

                        if category:
                            texture_name = os.path.basename(img.filepath) if img.filepath else img.name

                            large_textures.append({
                                "issue": "large_texture",
                                "material": mat.name,
                                "texture": texture_name,
                                "name": texture_name,
                                "size": f"{width}x{height}",
                                "category": category,
                                "technical_detail": f"Texture resolution {width}x{height} exceeds {category} threshold",
                                "user_message": f"The texture '{texture_name}' is {category} resolution ({width}x{height}). Large textures increase render time and memory usage, potentially causing out-of-memory errors on the render farm.",
                                "fix_instructions": "Consider downscaling the texture if the detail isn't needed at this resolution. In image editing software, resize to 2K (2048x2048) or 1K (1024x1024) if appropriate for your scene scale.",
                                "can_auto_fix": False
                            })

                            processed_images.add(img_id)
            except Exception:
                continue
    except Exception:
        pass

    try:
        # Check world shader nodes
        for world in bpy.data.worlds:
            if not world.use_nodes or not world.node_tree:
                continue

            try:
                for node in world.node_tree.nodes:
                    if node.type == 'TEX_IMAGE' and node.image:
                        img = node.image
                        img_id = img.name

                        if img_id in processed_images:
                            continue

                        width = img.size[0]
                        height = img.size[1]

                        category = None
                        if width >= 8192 or height >= 8192:
                            category = "8K"
                        elif width >= 4096 or height >= 4096:
                            category = "4K"

                        if category:
                            texture_name = os.path.basename(img.filepath) if img.filepath else img.name

                            large_textures.append({
                                "issue": "large_texture",
                                "material": f"World-{world.name}",
                                "texture": texture_name,
                                "name": texture_name,
                                "size": f"{width}x{height}",
                                "category": category,
                                "technical_detail": f"World texture resolution {width}x{height} exceeds {category} threshold",
                                "user_message": f"The environment texture '{texture_name}' is {category} resolution ({width}x{height}). This may slow down renders and consume excessive memory.",
                                "fix_instructions": "Consider downscaling the HDRI/environment map to 4K or 2K resolution, which is usually sufficient for background imagery.",
                                "can_auto_fix": False
                            })

                            processed_images.add(img_id)
            except Exception:
                continue
    except Exception:
        pass

    return large_textures


def check_zero_scale_objects():
    """
    Check for mesh objects with zero or near-zero scale/dimensions - MEDIUM.

    These "ghost" objects are invisible but still processed during rendering,
    wasting resources. They often result from accidental scaling operations.

    Exclusion rules:
    - Only flags MESH objects (ignores EMPTY, LIGHT, CAMERA, SPEAKER, etc.)
    - Objects with children are NOT flagged (they may act as control empties)
    """
    zero_scale_objects = []

    try:
        for obj in bpy.data.objects:
            # Only check MESH objects
            if obj.type != 'MESH':
                continue

            # Skip objects that have children (they may be parent controllers)
            if len(obj.children) > 0:
                continue

            try:
                # Check scale - any axis near zero makes object invisible
                scale = obj.scale
                scale_is_zero = (
                    abs(scale.x) < ZERO_THRESHOLD or
                    abs(scale.y) < ZERO_THRESHOLD or
                    abs(scale.z) < ZERO_THRESHOLD
                )

                # Check dimensions - all dimensions near zero means collapsed mesh
                dims = obj.dimensions
                dims_are_zero = (
                    abs(dims.x) < ZERO_THRESHOLD and
                    abs(dims.y) < ZERO_THRESHOLD and
                    abs(dims.z) < ZERO_THRESHOLD
                )

                if scale_is_zero or dims_are_zero:
                    # Determine the reason
                    if scale_is_zero:
                        reason = f"scale ({scale.x:.4f}, {scale.y:.4f}, {scale.z:.4f})"
                    else:
                        reason = f"dimensions ({dims.x:.4f}, {dims.y:.4f}, {dims.z:.4f})"

                    zero_scale_objects.append({
                        "issue": "zero_scale_object",
                        "type": "Mesh",
                        "object": obj.name,
                        "name": obj.name,
                        "scale": f"({scale.x:.4f}, {scale.y:.4f}, {scale.z:.4f})",
                        "dimensions": f"({dims.x:.4f}, {dims.y:.4f}, {dims.z:.4f})",
                        "reason": reason,
                        "technical_detail": f"Mesh object '{obj.name}' has {reason} near zero",
                        "user_message": f"The mesh '{obj.name}' has {reason} near zero, making it invisible. This 'ghost' object still gets processed during rendering, wasting resources.",
                        "fix_instructions": f"Review object '{obj.name}' in the Outliner. If it's not needed, delete it. If it should be visible, reset its scale with Ctrl+A > Scale or manually adjust the scale values in the Properties panel.",
                        "can_auto_fix": False  # Destructive - requires manual review
                    })

            except Exception:
                continue

    except Exception:
        pass

    return zero_scale_objects


def check_unconnected_nodes():
    """Check for Image Texture nodes with no output connections - LOW."""
    unconnected_nodes = []

    try:
        # Check all materials
        for mat in bpy.data.materials:
            if not mat.use_nodes or not mat.node_tree:
                continue

            try:
                for node in mat.node_tree.nodes:
                    if node.type == 'TEX_IMAGE':
                        has_connection = False

                        for output in node.outputs:
                            if output.is_linked:
                                has_connection = True
                                break

                        if not has_connection:
                            texture_name = "No Image"
                            if node.image:
                                texture_name = os.path.basename(node.image.filepath) if node.image.filepath else node.image.name

                            unconnected_nodes.append({
                                "issue": "unconnected_texture_node",
                                "material": mat.name,
                                "name": texture_name,
                                "node": node.name if node.name else "Image Texture",
                                "texture": texture_name,
                                "technical_detail": f"Image Texture node '{node.name}' has no output connections in material '{mat.name}'",
                                "user_message": f"Material '{mat.name}' has an unused texture node for '{texture_name}'. While this won't break rendering, it clutters your node tree and wastes memory loading unused textures.",
                                "fix_instructions": f"Open the Shader Editor, select material '{mat.name}', and either connect the texture node to your shader or delete it if it's not needed.",
                                "can_auto_fix": False
                            })
            except Exception:
                continue
    except Exception:
        pass

    try:
        # Check world shader nodes
        for world in bpy.data.worlds:
            if not world.use_nodes or not world.node_tree:
                continue

            try:
                for node in world.node_tree.nodes:
                    if node.type == 'TEX_IMAGE':
                        has_connection = False

                        for output in node.outputs:
                            if output.is_linked:
                                has_connection = True
                                break

                        if not has_connection:
                            texture_name = "No Image"
                            if node.image:
                                texture_name = os.path.basename(node.image.filepath) if node.image.filepath else node.image.name

                            unconnected_nodes.append({
                                "issue": "unconnected_texture_node",
                                "material": f"World-{world.name}",
                                "name": texture_name,
                                "node": node.name if node.name else "Image Texture",
                                "texture": texture_name,
                                "technical_detail": f"Image Texture node '{node.name}' has no output connections in world '{world.name}'",
                                "user_message": f"The world shader has an unused texture node for '{texture_name}'. This wastes memory loading a texture that won't be used.",
                                "fix_instructions": "Open the Shader Editor in World mode and either connect or delete the unused texture node.",
                                "can_auto_fix": False
                            })
            except Exception:
                continue
    except Exception:
        pass

    return unconnected_nodes


def check_unapplied_modifiers():
    """Check for unapplied modifiers on mesh objects - LOW."""
    unapplied_modifiers = []

    try:
        for obj in bpy.data.objects:
            if obj.type != 'MESH':
                continue

            try:
                for modifier in obj.modifiers:
                    unapplied_modifiers.append({
                        "issue": "unapplied_modifier",
                        "object": obj.name,
                        "name": obj.name,
                        "modifier": modifier.name,
                        "type": modifier.type,
                        "technical_detail": f"Object '{obj.name}' has unapplied {modifier.type} modifier '{modifier.name}'",
                        "user_message": f"Object '{obj.name}' has an unapplied {modifier.type} modifier. Unapplied modifiers slightly increase render time as they're calculated per frame. This is usually fine, but applying them can improve performance.",
                        "fix_instructions": f"Select object '{obj.name}', go to the Modifiers panel, and click 'Apply' on the '{modifier.name}' modifier. Warning: This is permanent, so save a backup first if you need to adjust the modifier later.",
                        "can_auto_fix": False  # Destructive - requires manual review
                    })
            except Exception:
                continue
    except Exception:
        pass

    return unapplied_modifiers


# -----------------------------------------------------------------------------
# Main Validation Runner
# -----------------------------------------------------------------------------

def determine_validation_status(critical, high, medium, low):
    """Determine overall validation status based on issues found."""
    if critical:
        return "FAIL"
    elif high:
        return "FAIL"
    elif medium:
        return "WARNING"
    elif low:
        return "WARNING"
    else:
        return "PASS"


def run_all_checks():
    """Run all validation checks and return results as a dictionary."""
    # Run all checks
    unpacked_items = check_packed_status()
    missing_textures = check_missing_textures()
    psd_files = check_psd_files()
    duplicate_images = check_duplicate_images()
    large_textures = check_large_textures()
    duplicate_materials = check_duplicate_materials()
    zero_scale_objects = check_zero_scale_objects()
    unconnected_nodes = check_unconnected_nodes()
    unapplied_modifiers = check_unapplied_modifiers()

    # Separate material duplicates by type
    identical_materials = [d for d in duplicate_materials if d.get('is_identical', False)]
    conflict_materials = [d for d in duplicate_materials if not d.get('is_identical', True)]

    # Organize issues by severity
    critical_issues = unpacked_items + missing_textures + psd_files
    high_issues = conflict_materials + duplicate_images  # Naming conflicts and duplicate images are high priority
    medium_issues = large_textures + identical_materials + zero_scale_objects
    low_issues = unconnected_nodes + unapplied_modifiers

    # Determine validation status
    validation_status = determine_validation_status(
        critical_issues, high_issues, medium_issues, low_issues
    )

    # Build result
    result = {
        "validation_status": validation_status,
        "timestamp": datetime.utcnow().isoformat() + "Z",
        "scene_info": get_scene_info(),
        "summary": {
            "critical": len(critical_issues),
            "high": len(high_issues),
            "medium": len(medium_issues),
            "low": len(low_issues),
            "total_issues": len(critical_issues) + len(high_issues) + len(medium_issues) + len(low_issues)
        },
        "issues": {
            "critical": critical_issues,
            "high": high_issues,
            "medium": medium_issues,
            "low": low_issues
        }
    }

    return result
